/*
 * Decompiled with CFR 0.152.
 */
package cz.xtf.jms;

import java.io.Serializable;
import java.net.SocketException;
import java.util.UUID;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import javax.jms.Connection;
import javax.jms.ConnectionFactory;
import javax.jms.Destination;
import javax.jms.ExceptionListener;
import javax.jms.JMSException;
import javax.jms.Message;
import javax.jms.MessageConsumer;
import javax.jms.MessageProducer;
import javax.jms.Session;
import javax.jms.TextMessage;
import javax.jms.Topic;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class JmsClient
implements AutoCloseable {
    private static final Logger LOGGER = LoggerFactory.getLogger(JmsClient.class);
    private static final ExecutorService EXECUTOR = Executors.newCachedThreadPool();
    private final long RECEIVE_TIMEOUT = 5000L;
    private ConnectionFactory factory;
    private Connection liveConnection;
    private Connection topicConnection;
    private String destinationName;
    private boolean isQueue;
    private boolean isPersistant = false;
    private boolean isTransacted = false;
    private boolean isDurable;
    private boolean keepAlive = false;
    private long timeToLive = 0L;
    private int retries = 10;

    public JmsClient(ConnectionFactory factory) {
        this.factory = factory;
    }

    public JmsClient(Connection connection) {
        this.keepAlive = true;
        this.liveConnection = connection;
    }

    public JmsClient addQueue(String name) {
        if (this.destinationName != null) {
            throw new IllegalArgumentException("Can't set more than one destination per client");
        }
        this.destinationName = name;
        this.isQueue = true;
        return this;
    }

    public JmsClient addTopic(String name) {
        if (this.destinationName != null) {
            throw new IllegalArgumentException("Can't set more than one destination per client");
        }
        this.destinationName = name;
        this.isQueue = false;
        return this;
    }

    public JmsClient persistant() {
        this.isPersistant = true;
        return this;
    }

    public JmsClient transacted() {
        this.isTransacted = true;
        return this;
    }

    public JmsClient durable() {
        this.isDurable = true;
        return this;
    }

    public JmsClient setRetries(int retries) {
        this.retries = retries;
        return this;
    }

    public MessageConsumer createTopicConsumer() throws JMSException {
        return this.createTopicConsumer(null);
    }

    public MessageConsumer createTopicConsumer(String selector) throws JMSException {
        if (this.isQueue) {
            throw new IllegalArgumentException("Only for topic, not queue");
        }
        String consumerId = "consumer-" + UUID.randomUUID();
        this.topicConnection = this.startConnection(consumerId);
        Session session = this.topicConnection.createSession(this.isTransacted, 1);
        Topic topic = session.createTopic(this.destinationName);
        if (this.isDurable) {
            if (selector != null) {
                return session.createDurableSubscriber(topic, consumerId, selector, true);
            }
            return session.createDurableSubscriber(topic, consumerId);
        }
        if (selector != null) {
            return session.createConsumer((Destination)topic, selector);
        }
        return session.createConsumer((Destination)topic);
    }

    public JmsClient keepAlive() {
        LOGGER.warn("When keepAlive is used, connection must be closed manually");
        this.keepAlive = true;
        return this;
    }

    public JmsClient timeToLive(long timeToLive) {
        this.timeToLive = timeToLive;
        return this;
    }

    public Message createMessage() throws JMSException {
        return this.createMessage(null);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public Message createMessage(Object messageObject) throws JMSException {
        Connection connection = null;
        Object result = null;
        try {
            connection = this.startConnection();
            try (Session session = null;){
                session = connection.createSession(this.isTransacted, 1);
                result = messageObject == null ? session.createMessage() : (messageObject instanceof String ? session.createTextMessage((String)messageObject) : session.createObjectMessage((Serializable)messageObject));
            }
        }
        finally {
            this.safeCloseConnection(connection);
        }
        return result;
    }

    public void sendMessage() throws JMSException {
        this.sendMessage("Hello, world!");
    }

    public void sendMessage(String messageText) throws JMSException {
        this.sendMessage(this.createMessage(messageText));
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void sendMessage(Message message) throws JMSException {
        Connection connection = null;
        try {
            connection = this.startConnection();
            try (Session session = null;){
                session = connection.createSession(this.isTransacted, 1);
                Object dest = this.isQueue ? session.createQueue(this.destinationName) : session.createTopic(this.destinationName);
                try (MessageProducer producer = session.createProducer((Destination)dest);){
                    if (this.isPersistant) {
                        producer.setDeliveryMode(2);
                    }
                    if (this.timeToLive > 0L) {
                        producer.setTimeToLive(this.timeToLive);
                    }
                    producer.send(message);
                }
            }
        }
        finally {
            this.safeCloseConnection(connection);
        }
    }

    public Message receiveMessage() throws JMSException {
        return this.receiveMessage(5000L, null);
    }

    public Message receiveMessage(String selector) throws JMSException {
        return this.receiveMessage(5000L, selector);
    }

    public Message receiveMessage(long timeout) throws JMSException {
        return this.receiveMessage(timeout, null);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public Message receiveMessage(long timeout, String selector) throws JMSException {
        Connection connection = null;
        Message result = null;
        try {
            connection = this.startConnection();
            try (Session session = null;){
                session = connection.createSession(this.isTransacted, 1);
                Object dest = this.isQueue ? session.createQueue(this.destinationName) : session.createTopic(this.destinationName);
                try (MessageConsumer consumer = selector != null ? session.createConsumer((Destination)dest, selector) : session.createConsumer((Destination)dest);){
                    result = consumer.receive(timeout);
                }
            }
        }
        finally {
            this.safeCloseConnection(connection);
        }
        return result;
    }

    public static String getTextMessage(Message message) throws JMSException {
        if (message != null && message instanceof TextMessage) {
            return ((TextMessage)message).getText();
        }
        return null;
    }

    public void disconnect() {
        if (this.keepAlive && this.liveConnection != null) {
            this.safeCloseConnection(this.liveConnection);
        }
        if (this.topicConnection != null) {
            this.safeCloseConnection(this.topicConnection);
        }
    }

    @Override
    public void close() throws Exception {
        this.disconnect();
    }

    private Connection createConnection() throws JMSException {
        Connection connection;
        if (this.destinationName == null) {
            throw new IllegalArgumentException("Destination is null, can't send message to nowhere");
        }
        if (this.keepAlive) {
            if (this.liveConnection == null) {
                this.liveConnection = this.factory.createConnection();
            }
            connection = this.liveConnection;
        } else {
            connection = this.factory.createConnection();
        }
        return connection;
    }

    private void safeCloseConnection(Connection connection) {
        try {
            if (connection != null) {
                connection.stop();
                if (!this.keepAlive) {
                    connection.close();
                }
            }
        }
        catch (JMSException e) {
            LOGGER.debug("Error while disconnecting", (Throwable)e);
        }
    }

    private Connection startConnection() throws JMSException {
        return this.startConnection(null);
    }

    private Connection startConnection(String consumerId) throws JMSException {
        Connection connection = null;
        int attempts = this.retries;
        while (connection == null && attempts > 0) {
            try {
                connection = this.createConnection();
                if (!this.isQueue && consumerId != null || this.keepAlive) {
                    connection.setExceptionListener((ExceptionListener)new ReconnectListener());
                }
                if (consumerId != null) {
                    connection.setClientID(consumerId);
                }
                Future<Void> future = EXECUTOR.submit(new StartConnection(connection));
                future.get(15L, TimeUnit.SECONDS);
            }
            catch (InterruptedException ex) {
                LOGGER.warn("Interrupted while starting connection", (Throwable)ex);
            }
            catch (ExecutionException ex) {
                LOGGER.warn("Error during connection start, reattempt");
                LOGGER.debug("Exception: ", (Throwable)ex);
                connection = null;
                --attempts;
                try {
                    Thread.sleep(10000L);
                }
                catch (InterruptedException e) {
                    LOGGER.error("Failed to start connection, {} attempts remaining", (Object)attempts, (Object)e);
                }
            }
            catch (TimeoutException ex) {
                this.safeCloseConnection(connection);
                connection = null;
                LOGGER.error("Failed to start connection, {} attempts remaining", (Object)(--attempts));
            }
            catch (JMSException ex) {
                if (ex.getCause() instanceof SocketException || ex.getMessage().contains("Connection reset")) {
                    LOGGER.warn("SocketException during connection start");
                    LOGGER.debug("Exception: ", (Throwable)ex);
                    connection = null;
                    --attempts;
                    try {
                        Thread.sleep(10000L);
                    }
                    catch (InterruptedException e) {
                        LOGGER.error("Failed to start connection, {} attempts remaining", (Object)attempts, (Object)e);
                    }
                    continue;
                }
                throw ex;
            }
        }
        if (connection == null) {
            throw new JMSException("Unable to start connection, see logs for errors.");
        }
        return connection;
    }

    private class ReconnectListener
    implements ExceptionListener {
        private int retries = 3;

        private ReconnectListener() {
        }

        public void onException(JMSException e) {
            LOGGER.debug("ExceptionListener invoked");
            try {
                if (this.retries > 0) {
                    LOGGER.debug("Attempting to reconnect, retries left {}", (Object)this.retries);
                    --this.retries;
                    if (JmsClient.this.topicConnection != null && !JmsClient.this.isQueue) {
                        JmsClient.this.topicConnection.start();
                    } else if (JmsClient.this.liveConnection != null) {
                        JmsClient.this.liveConnection.start();
                    }
                } else {
                    LOGGER.debug("Unable to reconnect", (Throwable)e);
                }
            }
            catch (JMSException ex) {
                LOGGER.debug("Exception thrown in ExceptionListener reconnect", (Throwable)ex);
            }
        }
    }

    private static class StartConnection
    implements Callable<Void> {
        private final Connection connection;

        public StartConnection(Connection connection) {
            this.connection = connection;
        }

        @Override
        public Void call() throws JMSException {
            this.connection.start();
            return null;
        }
    }
}

